<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Mretur extends CI_Model {

	/**
	 * Get all retur with customer and transaction details
	 */
	public function get_all()
	{
		// Cek kolom yang ada di tabel transaksi
		$has_order_id = $this->db->field_exists('order_id', 'transaksi');
		$has_total = $this->db->field_exists('total', 'transaksi');
		$has_total_harga = $this->db->field_exists('total_harga', 'transaksi');
		
		// Build select statement
		$select = 'r.*, c.nama_customer, c.email_customer';
		
		// Add order_id or fallback to id_transaksi
		if($has_order_id) {
			$select .= ', t.order_id';
		} else {
			$select .= ', t.id_transaksi as order_id';
		}
		
		// Add total or fallback to total_harga or null
		if($has_total) {
			$select .= ', t.total';
		} elseif($has_total_harga) {
			$select .= ', t.total_harga as total';
		} else {
			$select .= ', 0 as total';
		}
		
		$this->db->select($select);
		$this->db->from('retur r');
		$this->db->join('customer c', 'c.id_customer = r.id_customer', 'left');
		$this->db->join('transaksi t', 't.id_transaksi = r.id_transaksi', 'left');
		$this->db->order_by('r.id_retur', 'DESC');
		return $this->db->get();
	}

	/**
	 * Get retur by ID
	 */
	public function get_by_id($id_retur)
	{
		// Cek kolom yang ada di tabel transaksi
		$has_order_id = $this->db->field_exists('order_id', 'transaksi');
		$has_total = $this->db->field_exists('total', 'transaksi');
		$has_total_harga = $this->db->field_exists('total_harga', 'transaksi');
		$has_alamat = $this->db->field_exists('alamat_pengiriman', 'transaksi');
		$has_telepon = $this->db->field_exists('telepon_pengiriman', 'transaksi');
		
		// Build select statement
		$select = 'r.*, c.nama_customer, c.email_customer, c.telp_customer';
		
		// Add order_id
		if($has_order_id) {
			$select .= ', t.order_id';
		} else {
			$select .= ', t.id_transaksi as order_id';
		}
		
		// Add total
		if($has_total) {
			$select .= ', t.total';
		} elseif($has_total_harga) {
			$select .= ', t.total_harga as total';
		} else {
			$select .= ', 0 as total';
		}
		
		// Add alamat_pengiriman
		if($has_alamat) {
			$select .= ', t.alamat_pengiriman';
		} else {
			$select .= ', "" as alamat_pengiriman';
		}
		
		// Add telepon_pengiriman
		if($has_telepon) {
			$select .= ', t.telepon_pengiriman';
		} else {
			$select .= ', "" as telepon_pengiriman';
		}
		
		$this->db->select($select);
		$this->db->from('retur r');
		$this->db->join('customer c', 'c.id_customer = r.id_customer', 'left');
		$this->db->join('transaksi t', 't.id_transaksi = r.id_transaksi', 'left');
		$this->db->where('r.id_retur', $id_retur);
		return $this->db->get();
	}

	/**
	 * Get retur by customer ID
	 */
	public function get_by_customer($id_customer)
	{
		// Cek kolom yang ada di tabel transaksi
		$has_order_id = $this->db->field_exists('order_id', 'transaksi');
		$has_total = $this->db->field_exists('total', 'transaksi');
		$has_total_harga = $this->db->field_exists('total_harga', 'transaksi');
		
		// Build select statement
		$select = 'r.*';
		
		// Add order_id
		if($has_order_id) {
			$select .= ', t.order_id';
		} else {
			$select .= ', t.id_transaksi as order_id';
		}
		
		// Add total
		if($has_total) {
			$select .= ', t.total';
		} elseif($has_total_harga) {
			$select .= ', t.total_harga as total';
		} else {
			$select .= ', 0 as total';
		}
		
		$this->db->select($select);
		$this->db->from('retur r');
		$this->db->join('transaksi t', 't.id_transaksi = r.id_transaksi', 'left');
		$this->db->where('r.id_customer', $id_customer);
		$this->db->order_by('r.id_retur', 'DESC');
		return $this->db->get();
	}

	/**
	 * Get retur by transaction ID
	 */
	public function get_by_transaksi($id_transaksi)
	{
		$this->db->where('id_transaksi', $id_transaksi);
		return $this->db->get('retur');
	}

	/**
	 * Insert new retur
	 */
	public function insert($data)
	{
		// Set default created_at jika belum ada
		if(!isset($data['created_at']))
		{
			date_default_timezone_set('Asia/Jakarta');
			$data['created_at'] = date('Y-m-d H:i:s');
		}

		// Set default status jika belum ada
		if(!isset($data['status_retur']))
		{
			$data['status_retur'] = 'pending';
		}

		return $this->db->insert('retur', $data);
	}

	/**
	 * Update retur
	 */
	public function update($id_retur, $data)
	{
		$this->db->where('id_retur', $id_retur);
		return $this->db->update('retur', $data);
	}

	/**
	 * Update status retur
	 */
	public function update_status($id_retur, $status, $catatan_admin = null)
	{
		$data = array(
			'status_retur' => $status,
			'updated_at' => date('Y-m-d H:i:s')
		);

		if($catatan_admin) {
			$data['catatan_admin'] = $catatan_admin;
		}

		$this->db->where('id_retur', $id_retur);
		return $this->db->update('retur', $data);
	}

	/**
	 * Delete retur
	 */
	public function delete($id_retur)
	{
		$this->db->where('id_retur', $id_retur);
		return $this->db->delete('retur');
	}

	/**
	 * Count retur by status
	 */
	public function count_by_status($status)
	{
		$this->db->where('status_retur', $status);
		return $this->db->count_all_results('retur');
	}

	/**
	 * Get statistics
	 */
	public function get_statistics()
	{
		$this->db->select('
			COUNT(*) as total_retur,
			SUM(CASE WHEN status_retur = "pending" THEN 1 ELSE 0 END) as retur_pending,
			SUM(CASE WHEN status_retur = "disetujui" THEN 1 ELSE 0 END) as retur_disetujui,
			SUM(CASE WHEN status_retur = "ditolak" THEN 1 ELSE 0 END) as retur_ditolak,
			SUM(CASE WHEN status_retur = "selesai" THEN 1 ELSE 0 END) as retur_selesai
		', FALSE);
		return $this->db->get('retur')->row();
	}

	/**
	 * Get retur items (products in retur)
	 */
	public function get_retur_items($id_retur)
	{
		$this->db->select('ri.*, p.nama_produk, p.harga_produk, p.gambar_produk');
		$this->db->from('retur_item ri');
		$this->db->join('produk p', 'p.id_produk = ri.id_produk', 'left');
		$this->db->where('ri.id_retur', $id_retur);
		return $this->db->get();
	}

	/**
	 * Insert retur item
	 */
	public function insert_item($data)
	{
		return $this->db->insert('retur_item', $data);
	}
}

