<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Checkout extends CI_Controller {

	public function __construct()
	{
		parent::__construct();
		$this->load->model('Mkeranjang');
		$this->load->model('Mtransaksi');
		$this->load->model('Mmember');
		$this->load->model('Mongkir');
		$this->load->model('Mproduk');
		$this->load->library('Midtrans');
		$this->load->helper('url');
		$this->load->helper('indonesia_data');
		$this->load->helper('promo'); // Load promo helper
		$this->load->library('form_validation');
	}

	public function index()
	{
		// Set execution time limit untuk menghindari timeout
		set_time_limit(60);
		
		// Cek apakah user sudah login
		if(!$this->session->userdata('logged_in')) {
			$this->session->set_flashdata('error', 'Silakan login terlebih dahulu!');
			redirect('akun');
		}

		$data['title'] = 'Checkout - Marketplace';
		$data['nama_customer'] = $this->session->userdata('nama_customer');
		$data['logged_in'] = $this->session->userdata('logged_in');
		
		// Ambil data customer
		$id_customer = $this->session->userdata('id_customer');
		$customer_query = $this->Mmember->get_by_id($id_customer);
		$data['customer'] = $customer_query->num_rows() > 0 ? $customer_query->row() : null;
		
		// Ambil data keranjang
		$keranjang_query = $this->Mkeranjang->get_by_customer($id_customer);
		$all_keranjang = $keranjang_query->num_rows() > 0 ? $keranjang_query->result() : array();
		
		// Cek apakah ini buy_now (langsung checkout tanpa ke keranjang)
		$buy_now_item = $this->session->userdata('buy_now_item');
		if($buy_now_item && $this->input->get('buy_now') == '1') {
			// Filter hanya item yang dipilih untuk buy_now
			$data['keranjang'] = array_values(array_filter($all_keranjang, function($item) use ($buy_now_item) {
				return $item->id_keranjang == $buy_now_item;
			}));
			// Hapus session buy_now_item setelah digunakan
			$this->session->unset_userdata('buy_now_item');
		} else {
			// Filter berdasarkan item yang dipilih
			$selected_items = $this->input->post('selected_items');
			if(!empty($selected_items)) {
				$data['keranjang'] = array_values(array_filter($all_keranjang, function($item) use ($selected_items) {
					return in_array($item->id_keranjang, $selected_items);
				}));
			} else {
				$data['keranjang'] = $all_keranjang;
			}
		}
		
		// Jika keranjang kosong, redirect ke keranjang
		if(empty($data['keranjang'])) {
			$this->session->set_flashdata('error', 'Pilih produk yang ingin di-checkout!');
			redirect('keranjang');
		}
		
		// Hitung total dengan diskon promo
		$data['subtotal'] = 0;
		foreach($data['keranjang'] as $item) {
			// Tentukan harga awal (prioritas harga ukuran, lalu harga_produk dari tabel produk, fallback ke harga di keranjang)
			$harga_awal = 0;
			
			// Cek apakah ada harga ukuran (dari join tabel produk_ukuran)
			if(isset($item->harga_ukuran) && $item->harga_ukuran > 0) {
				$harga_awal = $item->harga_ukuran;
			} 
			// Cek apakah item memiliki ukuran tapi harga_ukuran tidak ter-set (gunakan harga dari keranjang yang sudah tersimpan saat add to cart)
			else if(isset($item->ukuran) && !empty($item->ukuran) && isset($item->harga) && $item->harga > 0) {
				$harga_awal = $item->harga;
			}
			// Jika tidak ada ukuran, gunakan harga produk (base price)
			else if(isset($item->harga_produk) && $item->harga_produk > 0) {
				$harga_awal = $item->harga_produk;
			} 
			// Fallback terakhir
			else if(isset($item->harga) && $item->harga > 0) {
				$harga_awal = $item->harga;
			}
			
			// Hitung diskon
			$promo_info = get_harga_diskon($item->id_produk, $harga_awal);
			
			// Update data item untuk view
			$item->harga_asli = $harga_awal; // Simpan harga asli untuk display coret
			$item->harga = $promo_info['harga_diskon']; // Override harga dengan harga diskon
			$item->diskon_persen = $promo_info['persen_diskon'];
			
			$qty = isset($item->qty) ? $item->qty : 1;
			$data['subtotal'] += $item->harga * $qty;
		}
		
		// Ongkir (default 0, akan dihitung via AJAX)
		$data['ongkir'] = $this->session->userdata('ongkir') ? $this->session->userdata('ongkir') : 0;
		$data['total'] = $data['subtotal'] + $data['ongkir'];
		
		// Ambil daftar provinsi dari Raja Ongkir via Model (dengan fallback ke data statis)
		try {
			$provinsi_result = $this->Mongkir->get_all_provinsi();

			if($provinsi_result !== false && is_array($provinsi_result) && count($provinsi_result) > 0) {
				// Berhasil mengambil data provinsi (dari API atau fallback statis)
				$data['provinsi'] = $provinsi_result;
				$data['rajaongkir_error'] = false;
				$data['use_manual_ongkir'] = false; // API berfungsi, bisa hitung otomatis
				log_message('info', 'Checkout: Berhasil mengambil ' . count($provinsi_result) . ' provinsi');
			} else {
				// Fallback ke data statis
				$data['provinsi'] = get_provinsi_static();
				$data['rajaongkir_error'] = false;
				$data['use_manual_ongkir'] = true; // Gunakan input manual
				log_message('info', 'Checkout: Menggunakan data provinsi statis');
			}
		} catch (Exception $e) {
			// Jika semua gagal, gunakan data statis
			$errMsg = $e->getMessage();
			log_message('info', 'Checkout: API gagal, menggunakan data statis. Error: ' . $errMsg);
			
			$data['provinsi'] = get_provinsi_static();
			$data['rajaongkir_error'] = false;
			$data['use_manual_ongkir'] = true; // Gunakan input manual
		}
		
		// ID kota asal (ganti dengan ID kota toko/warehouse Anda)
		// Contoh: Jakarta Pusat = 152
		$data['origin_city_id'] = 152; // Ganti dengan ID kota asal Anda
		
		// Pass Midtrans config ke view
		$data['midtrans_client_key'] = $this->midtrans->get_client_key();
		$data['midtrans_is_production'] = $this->midtrans->is_production();
		
		$this->load->view('header', $data);
		$this->load->view('checkout', $data);
		$this->load->view('footer');
	}
	
	public function get_kota()
	{
		$id_provinsi = $this->input->get('provinsi');
		
		header('Content-Type: application/json');
		
		if($id_provinsi) {
			try {
				$kota = $this->Mongkir->get_kota_by_provinsi($id_provinsi);
				if (!empty($kota)) {
					echo json_encode(array('status' => 'success', 'data' => $kota));
				} else {
					// Fallback ke data statis jika API mengembalikan kosong
					$kota_static = get_kota_by_provinsi_static($id_provinsi);
					if (!empty($kota_static)) {
						echo json_encode(array('status' => 'success', 'data' => $kota_static));
					} else {
						echo json_encode(array('status' => 'error', 'message' => 'Data kota untuk provinsi ini tidak tersedia. Silakan hubungi admin.'));
					}
				}
			} catch (Exception $e) {
				log_message('info', 'Checkout.get_kota Exception: ' . $e->getMessage() . ' - Mencoba fallback ke data statis');
				// Fallback ke data statis
				$kota_static = get_kota_by_provinsi_static($id_provinsi);
				if (!empty($kota_static)) {
					echo json_encode(array('status' => 'success', 'data' => $kota_static));
				} else {
					echo json_encode(array('status' => 'error', 'message' => 'Gagal mengambil data kota: ' . $e->getMessage()));
				}
			}
		} else {
			echo json_encode(array('status' => 'error', 'message' => 'Provinsi tidak valid'));
		}
	}

	/**
	 * Endpoint AJAX untuk mengecek koneksi dan konfigurasi RajaOngkir secara cepat.
	 * Mengembalikan JSON {status: 'ok'|'error', message: '...'}
	 */
	public function check_rajaongkir()
	{
		header('Content-Type: application/json');
		try {
			$provinsi = $this->Mongkir->get_all_provinsi();
			if (is_array($provinsi) && count($provinsi) > 0) {
				echo json_encode(['status' => 'ok', 'message' => 'Berhasil terhubung ke RajaOngkir. Jumlah provinsi: ' . count($provinsi)]);
			} else {
				echo json_encode(['status' => 'error', 'message' => 'Response kosong dari RajaOngkir']);
			}
		} catch (Exception $e) {
			log_message('error', 'Checkout.check_rajaongkir Exception: ' . $e->getMessage());
			echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
		}
	}
	
	public function hitung_ongkir()
	{
		// Set execution time limit untuk menghindari timeout
		set_time_limit(30);
		
		$origin = $this->input->post('origin');
		$destination = $this->input->post('destination');
		$weight = $this->input->post('weight');
		$courier = $this->input->post('courier');
		
		header('Content-Type: application/json');
		
		// Validasi input
		if(!$this->Mongkir->validasi_ongkir($origin, $destination, $weight, $courier)) {
			echo json_encode(array('status' => 'error', 'message' => 'Data tidak lengkap atau tidak valid'));
			return;
		}
		
		// Jika berat kosong, default 1000 gram (1 kg)
		if(empty($weight) || $weight < 1) {
			$weight = 1000;
		}
		
		try {
			$result = $this->Mongkir->hitung_ongkir($origin, $destination, $weight, $courier);
			if($result && is_array($result) && count($result) > 0) {
				$courier_name = isset($result[0]['courier']) ? $result[0]['courier'] : strtoupper($courier);
				echo json_encode(array(
					'status' => 'success', 
					'data' => $result, 
					'courier' => $courier_name
				));
				return;
			}
			// jika hasil kosong
			echo json_encode(array('status' => 'error', 'message' => 'Hasil ongkir kosong')); 
		} catch (Exception $e) {
			log_message('info', 'Checkout.hitung_ongkir Exception: ' . $e->getMessage());
			
			// Cek jika error karena API endpoint tidak aktif (HTTP 410)
			$error_msg = $e->getMessage();
			if (stripos($error_msg, 'HTTP 410') !== false || stripos($error_msg, 'tidak aktif') !== false || stripos($error_msg, 'endpoint') !== false) {
				// API tidak tersedia, kembalikan response untuk mode manual
				echo json_encode(array(
					'status' => 'api_unavailable', 
					'message' => 'API Raja Ongkir tidak tersedia. Silakan masukkan ongkir secara manual.',
					'manual_mode' => true
				));
			} else {
				// Error lainnya
				echo json_encode(array(
					'status' => 'error', 
					'message' => 'Gagal menghitung ongkir. Silakan masukkan ongkir secara manual.',
					'manual_mode' => true
				));
			}
		}
	}

	public function proses()
	{
		// Set execution time limit
		set_time_limit(60);
		
		// Set header untuk JSON response (untuk AJAX)
		$is_ajax = $this->input->is_ajax_request() || 
		           (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') ||
		           (isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false);
		
		// Wrap seluruh proses dalam try-catch untuk menangkap semua error
		try {
			// Cek apakah user sudah login
			if(!$this->session->userdata('logged_in')) {
				if($is_ajax) {
					if (ob_get_level()) {
						ob_clean();
					}
					header('Content-Type: application/json; charset=utf-8');
					echo json_encode(array('status' => 'error', 'message' => 'Silakan login terlebih dahulu!'));
					exit;
				}
				$this->session->set_flashdata('error', 'Silakan login terlebih dahulu!');
				redirect('akun');
				return;
			}

			$id_customer = $this->session->userdata('id_customer');
			
			// Validasi input
			$this->form_validation->set_rules('alamat_pengiriman', 'Alamat Pengiriman', 'required|trim|min_length[10]');
			$this->form_validation->set_rules('telepon_pengiriman', 'Telepon Pengiriman', 'required|trim|numeric|min_length[10]');
			$this->form_validation->set_rules('provinsi', 'Provinsi', 'required');
			$this->form_validation->set_rules('kota', 'Kota/Kabupaten', 'required');
			
			// Set default metode pembayaran ke midtrans jika tidak ada
			if(empty($this->input->post('metode_pembayaran'))) {
				$_POST['metode_pembayaran'] = 'midtrans';
			}
			
			// Ongkir bisa dari input manual atau dari perhitungan otomatis
			$ongkir_manual = $this->input->post('ongkir_manual');
			$ongkir_auto = $this->input->post('ongkir');
			$kurir = $this->input->post('kurir');
			
			if (!empty($ongkir_manual) || $kurir == 'manual') {
				// Mode manual: ongkir manual wajib, kurir tidak wajib
				$this->form_validation->set_rules('ongkir_manual', 'Ongkos Kirim', 'required|numeric');
			} else {
				// Mode auto: kurir dan ongkir auto wajib
				$this->form_validation->set_rules('kurir', 'Kurir', 'required');
				$this->form_validation->set_rules('ongkir', 'Ongkos Kirim', 'required|numeric');
			}
			
			if($this->form_validation->run() == FALSE) {
				// Jika request AJAX, return JSON
				if($is_ajax) {
					if (ob_get_level()) {
						ob_clean();
					}
					header('Content-Type: application/json; charset=utf-8');
					$errors = validation_errors();
					// Bersihkan HTML tags dari error message
					$errors = strip_tags($errors);
					$errors = str_replace(array("\n", "\r"), ' ', $errors);
					echo json_encode(array('status' => 'error', 'message' => trim($errors)));
					exit;
				}
				
				$this->session->set_flashdata('error', validation_errors());
				redirect('checkout');
				return;
			}

			// Ambil data keranjang
			$keranjang_query = $this->Mkeranjang->get_by_customer($id_customer);
			$all_keranjang = $keranjang_query->num_rows() > 0 ? $keranjang_query->result() : array();
			
			// Filter berdasarkan item yang dipilih
			$selected_items = $this->input->post('selected_items');
			if(empty($selected_items)) {
				// Jika request AJAX, return JSON
				if($is_ajax) {
					if (ob_get_level()) {
						ob_clean();
					}
					header('Content-Type: application/json; charset=utf-8');
					echo json_encode(array('status' => 'error', 'message' => 'Pilih produk yang ingin di-checkout!'));
					exit;
				}
				
				$this->session->set_flashdata('error', 'Pilih produk yang ingin di-checkout!');
				redirect('keranjang');
				return;
			}
			$keranjang = array_values(array_filter($all_keranjang, function($item) use ($selected_items) {
				return in_array($item->id_keranjang, $selected_items);
			}));
			
			if(empty($keranjang)) {
				// Jika request AJAX, return JSON
				if($is_ajax) {
					if (ob_get_level()) {
						ob_clean();
					}
					header('Content-Type: application/json; charset=utf-8');
					echo json_encode(array('status' => 'error', 'message' => 'Item yang dipilih tidak ditemukan di keranjang!'));
					exit;
				}
				
				$this->session->set_flashdata('error', 'Item yang dipilih tidak ditemukan di keranjang!');
				redirect('keranjang');
				return;
			}

			// Validasi stok produk sebelum checkout
			$stok_error = array();
			foreach($keranjang as $item) {
				$produk = $this->Mproduk->get_by_id($item->id_produk);
				if($produk->num_rows() > 0) {
					$produk_data = $produk->row();
					$stok_produk = isset($produk_data->stok_produk) ? (int)$produk_data->stok_produk : 0;
					$qty_order = isset($item->qty) ? (int)$item->qty : 1;
					
					if($stok_produk <= 0) {
						$stok_error[] = 'Produk "' . $produk_data->nama_produk . '" sedang kehabisan stok!';
					} else if($qty_order > $stok_produk) {
						$stok_error[] = 'Stok produk "' . $produk_data->nama_produk . '" tidak mencukupi! Stok tersedia: ' . $stok_produk . ', yang diminta: ' . $qty_order;
					}
				}
			}
			
			if(!empty($stok_error)) {
				// Jika request AJAX, return JSON
				if($is_ajax) {
					if (ob_get_level()) {
						ob_clean();
					}
					header('Content-Type: application/json; charset=utf-8');
					echo json_encode(array('status' => 'error', 'message' => implode('\\n', $stok_error)));
					exit;
				}
				
				$this->session->set_flashdata('error', implode('<br>', $stok_error));
				redirect('keranjang');
				return;
			}

			// Hitung total
			$subtotal = 0;
			foreach($keranjang as $item) {
				$harga = isset($item->harga) ? $item->harga : 0;
				$qty = isset($item->qty) ? $item->qty : 1;
				$subtotal += $harga * $qty;
			}
			
			// Ambil ongkir dari form (manual atau auto)
			$ongkir_manual = $this->input->post('ongkir_manual');
			$ongkir_auto = $this->input->post('ongkir');
			$ongkir = !empty($ongkir_manual) ? (float)$ongkir_manual : ((float)$ongkir_auto ?: 0);
			$total = $subtotal + $ongkir;
			
			// Ambil data customer
			$customer_query = $this->Mmember->get_by_id($id_customer);
			$customer = $customer_query->num_rows() > 0 ? $customer_query->row() : null;
			
			// Generate order_id untuk Midtrans
			$order_id = 'ORDER-' . time() . '-' . $id_customer;
			
			// Simpan transaksi
			// Berdasarkan struktur tabel transaksi yang sebenarnya:
			// - id_transaksi (AUTO_INCREMENT) - tidak perlu diisi
			// - id_customer
			// - kode_transaksi (varchar(50)) - gunakan order_id
			// - tanggal_transaksi (AUTO CURRENT_TIMESTAMP) - tidak perlu diisi
			// - total_harga (decimal(10,2))
			// - status_transaksi (enum) - gunakan 'pending'
			// - alamat_pengiriman (text)
			// - metode_pembayaran (varchar(50), nullable)
			// - bukti_pembayaran (varchar(255), nullable) - tidak perlu diisi
			// - resi_pengiriman (varchar(100), nullable) - tidak perlu diisi
			// - catatan (text, nullable) - bisa simpan info ongkir di sini
			// - created_at (AUTO CURRENT_TIMESTAMP) - tidak perlu diisi
			// - updated_at (AUTO CURRENT_TIMESTAMP ON UPDATE) - tidak perlu diisi
			
			$data_transaksi = array(
				'id_customer' => $id_customer,
				'kode_transaksi' => $order_id, // Gunakan order_id sebagai kode_transaksi
				'total_harga' => $total,
				'status_transaksi' => 'pending', // Gunakan 'pending' sesuai enum
				'alamat_pengiriman' => $this->input->post('alamat_pengiriman'),
				'metode_pembayaran' => $this->input->post('metode_pembayaran')
			);
			
			// Simpan info ongkir di catatan (karena tidak ada kolom ongkir)
			$catatan_parts = array();
			if ($ongkir > 0) {
				$catatan_parts[] = 'Ongkir: Rp ' . number_format($ongkir, 0, ',', '.');
			}
			$telepon = $this->input->post('telepon_pengiriman');
			if (!empty($telepon)) {
				$catatan_parts[] = 'Telp: ' . $telepon;
			}
			if (!empty($catatan_parts)) {
				$data_transaksi['catatan'] = implode(' | ', $catatan_parts);
			}
			
			// Start database transaction
			$this->db->trans_start();
			
			// Insert transaksi
			$insert_result = $this->db->insert('transaksi', $data_transaksi);
			if (!$insert_result) {
				$this->db->trans_rollback();
				$db_error = $this->db->error();
				log_message('error', 'Failed to insert transaksi: ' . print_r($db_error, true));
				$error_msg = 'Gagal menyimpan transaksi ke database';
				if (!empty($db_error['message'])) {
					$error_msg .= ': ' . $db_error['message'];
				}
				throw new Exception($error_msg);
			}
			$id_transaksi = $this->db->insert_id();
			
			if (empty($id_transaksi)) {
				$this->db->trans_rollback();
				throw new Exception('Gagal mendapatkan ID transaksi');
			}
			
			// Siapkan item details untuk Midtrans
			$item_details = array();
			
			// Cek apakah tabel transaksi_detail ada
			$has_transaksi_detail = $this->db->table_exists('transaksi_detail');
			
			// Insert transaksi detail (jika tabel ada)
			foreach($keranjang as $item) {
				$harga = isset($item->harga) && $item->harga > 0 ? $item->harga : 0;
				$qty = isset($item->qty) ? $item->qty : 1;
				$subtotal_item = $harga * $qty;
				
				// Hanya insert ke transaksi_detail jika tabel ada
				if ($has_transaksi_detail) {
					$data_detail = array(
						'id_transaksi' => $id_transaksi,
						'id_produk' => $item->id_produk,
						'id_ukuran' => isset($item->id_ukuran) ? $item->id_ukuran : 0,
						'ukuran' => isset($item->ukuran) ? $item->ukuran : '',
						'harga' => $harga,
						'qty' => $qty,
						'subtotal' => $subtotal_item
					);
					
					$insert_detail = $this->db->insert('transaksi_detail', $data_detail);
					if (!$insert_detail) {
						$this->db->trans_rollback();
						$db_error = $this->db->error();
						log_message('error', 'Failed to insert transaksi_detail: ' . print_r($db_error, true));
						throw new Exception('Gagal menyimpan detail transaksi: ' . $db_error['message']);
					}
				}
				
				// Ambil nama produk dari database jika tidak ada di keranjang
				$produk = $this->Mproduk->get_by_id($item->id_produk);
				$nama_produk = 'Produk';
				if($produk->num_rows() > 0) {
					$produk_data = $produk->row();
					$nama_produk = isset($produk_data->nama_produk) ? $produk_data->nama_produk : 'Produk';
					
					// Update Stok
					$id_ukuran = isset($item->id_ukuran) ? $item->id_ukuran : 0;
					
					if($id_ukuran > 0) {
						// Kurangi stok ukuran
						$ukuran_data = $this->db->get_where('produk_ukuran', ['id_ukuran' => $id_ukuran])->row();
						if($ukuran_data) {
							$stok_sekarang = isset($ukuran_data->stok) ? (int)$ukuran_data->stok : 0;
							$stok_baru = $stok_sekarang - $qty;
							if($stok_baru < 0) $stok_baru = 0;
							
							$this->db->where('id_ukuran', $id_ukuran);
							$this->db->update('produk_ukuran', array('stok' => $stok_baru));
						}
					} else {
						// Kurangi stok produk utama
						$stok_sekarang = isset($produk_data->stok_produk) ? (int)$produk_data->stok_produk : 0;
						$stok_baru = $stok_sekarang - $qty;
						if($stok_baru < 0) $stok_baru = 0;
						
						$this->db->where('id_produk', $item->id_produk);
						$this->db->update('produk', array('stok_produk' => $stok_baru));
					}
				} else if(isset($item->nama_produk)) {
					$nama_produk = $item->nama_produk;
				}
				
				// Tambahkan ukuran ke nama produk jika ada
				if(isset($item->ukuran) && !empty($item->ukuran)) {
					$nama_produk .= ' (' . $item->ukuran . ')';
				}
				
				// Tambahkan ke item details untuk Midtrans
				$item_details[] = array(
					'id' => isset($item->id_produk) ? (string)$item->id_produk : '',
					'price' => (int)$harga,
					'quantity' => (int)$qty,
					'name' => $nama_produk
				);
			}
			
			// Tambahkan ongkir sebagai item
			if($ongkir > 0) {
				$item_details[] = array(
					'id' => 'ONGKIR',
					'price' => (int)$ongkir,
					'quantity' => 1,
					'name' => 'Ongkos Kirim'
				);
			}
			
			// Validasi item_details tidak kosong
			if(empty($item_details)) {
				throw new Exception('Item details tidak boleh kosong');
			}
			
			// Validasi total amount
			$calculated_total = 0;
			foreach($item_details as $item) {
				$calculated_total += (int)$item['price'] * (int)$item['quantity'];
			}
			
			// Pastikan total sesuai (toleransi 1 rupiah untuk rounding)
			if(abs($calculated_total - $total) > 1) {
				log_message('warning', 'Total mismatch: calculated=' . $calculated_total . ', expected=' . $total);
				// Update total dengan calculated total
				$total = $calculated_total;
			}
			
			// Siapkan parameter untuk Midtrans
			$midtrans_params = array(
				'transaction_details' => array(
					'order_id' => $order_id,
					'gross_amount' => (int)$total
				),
				'item_details' => $item_details,
				'customer_details' => array(
					'first_name' => isset($customer->nama_customer) ? $customer->nama_customer : 'Customer',
					'email' => isset($customer->email_customer) && !empty($customer->email_customer) ? $customer->email_customer : 'customer@example.com',
					'phone' => $this->input->post('telepon_pengiriman'),
					'billing_address' => array(
						'first_name' => isset($customer->nama_customer) ? $customer->nama_customer : 'Customer',
						'address' => $this->input->post('alamat_pengiriman'),
						'phone' => $this->input->post('telepon_pengiriman'),
						'country_code' => 'IDN'
					),
					'shipping_address' => array(
						'first_name' => isset($customer->nama_customer) ? $customer->nama_customer : 'Customer',
						'address' => $this->input->post('alamat_pengiriman'),
						'phone' => $this->input->post('telepon_pengiriman'),
						'country_code' => 'IDN'
					)
				),
				// Aktifkan metode pembayaran yang umum digunakan di Indonesia
				'enabled_payments' => array(
					'credit_card',           // Kartu Kredit/Debit
					'bank_transfer',        // Transfer Bank (BCA, Mandiri, BNI, Permata)
					'bca_va',               // Virtual Account BCA
					'bni_va',               // Virtual Account BNI
					'bri_va',               // Virtual Account BRI
					'permata_va',           // Virtual Account Permata
					'other_va',             // Virtual Account Lainnya
					'gopay',                // GoPay
					'shopee_pay',           // ShopeePay
					'klikbca',              // KlikBCA
					'bca_klikbca',          // BCA KlikPay
					'cimb_clicks',          // CIMB Clicks
					'danamon_online',       // Danamon Online Banking
					'bri_epay',             // BRI E-Pay
					'indomaret',            // Indomaret
					'alfamart',             // Alfamart
					'akulaku'               // Akulaku
				),
				'callbacks' => array(
					'finish' => base_url('checkout/finish'),
					'unfinish' => base_url('checkout/unfinish'),
					'error' => base_url('checkout/error')
				)
			);
			
			// Dapatkan snap token dari Midtrans
			try {
				// Pastikan Midtrans library sudah ter-load
				if (!isset($this->midtrans) || !is_object($this->midtrans)) {
					throw new Exception('Midtrans library tidak ter-load dengan benar');
				}
				
				$snap_token = $this->midtrans->get_snap_token($midtrans_params);
				
				// Validasi snap_token
				if (empty($snap_token) || $snap_token === false) {
					log_message('error', 'Snap token is empty or false. Response from Midtrans may be invalid.');
					throw new Exception('Snap token kosong dari Midtrans. Pastikan konfigurasi Midtrans (Server Key dan Client Key) sudah benar.');
				}
			} catch (Exception $e) {
				$this->db->trans_rollback();
				log_message('error', 'Midtrans get_snap_token error: ' . $e->getMessage());
				log_message('error', 'Midtrans params: ' . print_r($midtrans_params, true));
				
				if($is_ajax) {
					if (ob_get_level()) {
						ob_clean();
					}
					header('Content-Type: application/json; charset=utf-8');
					$error_msg = 'Gagal membuat transaksi pembayaran.';
					// Selalu tampilkan error message untuk debugging
					$error_msg .= ' ' . $e->getMessage();
					echo json_encode(array('status' => 'error', 'message' => $error_msg));
					exit;
				}
				
				$this->session->set_flashdata('error', 'Gagal membuat transaksi pembayaran! ' . $e->getMessage());
				redirect('checkout');
				return;
			}
			
			if($snap_token) {
				// Simpan data item dan snap_token di catatan (karena tidak ada kolom snap_token dan tabel transaksi_detail mungkin tidak ada)
				$current_catatan = isset($data_transaksi['catatan']) ? $data_transaksi['catatan'] : '';
				
				// Siapkan data untuk disimpan di catatan
				$catatan_data = array();
				
				// Simpan item details sebagai JSON (selalu simpan sebagai backup, bahkan jika tabel transaksi_detail ada)
				$items_data = array();
				foreach($keranjang as $item) {
					$produk = $this->Mproduk->get_by_id($item->id_produk);
					$nama_produk = 'Produk';
					$gambar_produk = '';
					$id_kategori = 0;
					
					if($produk->num_rows() > 0) {
						$produk_data = $produk->row();
						$nama_produk = isset($produk_data->nama_produk) ? $produk_data->nama_produk : 'Produk';
						$gambar_produk = isset($produk_data->gambar_produk) ? $produk_data->gambar_produk : (isset($produk_data->foto_produk) ? $produk_data->foto_produk : '');
						$id_kategori = isset($produk_data->id_kategori) ? $produk_data->id_kategori : 0;
					}
					
					$items_data[] = array(
						'id_produk' => $item->id_produk,
						'nama_produk' => $nama_produk,
						'harga' => isset($item->harga) && $item->harga > 0 ? $item->harga : 0,
						'qty' => isset($item->qty) ? $item->qty : 1,
						'subtotal' => (isset($item->harga) && $item->harga > 0 ? $item->harga : 0) * (isset($item->qty) ? $item->qty : 1),
						'gambar_produk' => $gambar_produk,
						'id_kategori' => $id_kategori
					);
				}
				
				// Tambahkan ongkir sebagai item jika ada
				if($ongkir > 0) {
					$items_data[] = array(
						'id_produk' => 'ONGKIR',
						'nama_produk' => 'Ongkos Kirim',
						'harga' => $ongkir,
						'qty' => 1,
						'subtotal' => $ongkir,
						'gambar_produk' => '',
						'id_kategori' => 0
					);
				}
				
				$catatan_data['items'] = $items_data;
				
				// Simpan snap_token full
				$catatan_data['snap_token'] = $snap_token;
				
				// Encode JSON dengan flag untuk memastikan format yang benar
				$json_string = json_encode($catatan_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
				
				// Gabungkan dengan catatan yang sudah ada
				$catatan_parts = array();
				if (!empty($current_catatan)) {
					$catatan_parts[] = $current_catatan;
				}
				$catatan_parts[] = 'JSON_DATA:' . $json_string;
				
				$catatan_final = implode(' | ', $catatan_parts);
				
				// Log untuk debugging
				log_message('debug', 'Saving JSON data to catatan for transaksi ID: ' . $id_transaksi);
				log_message('debug', 'JSON string length: ' . strlen($json_string));
				log_message('debug', 'Items count: ' . count($items_data));
				
				$this->db->where('id_transaksi', $id_transaksi);
				$update_result = $this->db->update('transaksi', array('catatan' => $catatan_final));
				
				if(!$update_result) {
					log_message('error', 'Failed to update catatan for transaksi ID: ' . $id_transaksi);
				}
				
				// Hapus keranjang setelah checkout
				if(!empty($selected_items)) {
					$this->Mkeranjang->clear_selected($selected_items);
				} else {
					$this->Mkeranjang->clear($id_customer);
				}

				// Insert Notifikasi untuk Admin
				// Ambil nama customer jika belum ada
				$notif_nama_customer = 'Customer';
				if(isset($this->session->userdata['nama_customer'])) {
					$notif_nama_customer = $this->session->userdata['nama_customer'];
				} else {
					$cust_data = $this->db->get_where('customer', array('id_customer' => $id_customer))->row();
					if($cust_data) $notif_nama_customer = $cust_data->nama_customer;
				}

				$this->db->insert('notifikasi', array(
					'judul' => 'Transaksi Baru',
					'pesan' => 'Pesanan baru ' . $order_id . ' dari ' . $notif_nama_customer,
					'link' => 'transaksi/detail/' . $id_transaksi,
					'jenis' => 'transaksi',
					'is_read' => 0,
					'created_at' => date('Y-m-d H:i:s')
				));
				
				$this->db->trans_complete();
				
				if($this->db->trans_status() === FALSE) {
					// Jika request AJAX, return JSON
					if($is_ajax) {
						if (ob_get_level()) {
							ob_clean();
						}
						header('Content-Type: application/json; charset=utf-8');
						echo json_encode(array('status' => 'error', 'message' => 'Gagal memproses checkout! Silakan coba lagi.'));
						exit;
					}
					$this->session->set_flashdata('error', 'Gagal memproses checkout! Silakan coba lagi.');
					redirect('checkout');
					return;
				} else {
					// Jika request AJAX, return JSON dengan snap_token
					if($is_ajax) {
						// Pastikan tidak ada output sebelum JSON
						if (ob_get_level()) {
							ob_clean();
						}
						header('Content-Type: application/json; charset=utf-8');
						echo json_encode(array(
							'status' => 'success',
							'snap_token' => $snap_token,
							'order_id' => $order_id
						));
						exit;
					}
					
					// Simpan ongkir ke session untuk ditampilkan di payment
					$this->session->set_userdata('ongkir', $ongkir);
					
					// Redirect ke halaman pembayaran Midtrans (fallback untuk non-AJAX)
					$this->session->set_userdata('snap_token', $snap_token);
					$this->session->set_userdata('order_id', $order_id);
					redirect('checkout/payment');
					return;
				}
			} else {
				$this->db->trans_rollback();
				
				// Jika request AJAX, return JSON
				if($is_ajax) {
					if (ob_get_level()) {
						ob_clean();
					}
					header('Content-Type: application/json; charset=utf-8');
					echo json_encode(array('status' => 'error', 'message' => 'Gagal membuat transaksi pembayaran! Snap token tidak berhasil dibuat.'));
					exit;
				}
				
				$this->session->set_flashdata('error', 'Gagal membuat transaksi pembayaran! Silakan coba lagi.');
				redirect('checkout');
				return;
			}
		
		} catch (Exception $e) {
			// Tangkap semua error yang tidak tertangkap
			log_message('error', 'Checkout proses error: ' . $e->getMessage());
			log_message('error', 'Checkout proses trace: ' . $e->getTraceAsString());
			log_message('error', 'Checkout proses file: ' . $e->getFile() . ' line: ' . $e->getLine());
			
			// Rollback transaction jika ada
			try {
				if ($this->db->trans_status() !== FALSE) {
					$this->db->trans_rollback();
				}
			} catch (Exception $rollback_ex) {
				log_message('error', 'Failed to rollback: ' . $rollback_ex->getMessage());
			}
			
			if($is_ajax) {
				if (ob_get_level()) {
					ob_clean();
				}
				header('Content-Type: application/json; charset=utf-8');
				$error_message = 'Terjadi kesalahan saat memproses checkout.';
				// Di development, tampilkan error detail
				if (ENVIRONMENT === 'development' || ENVIRONMENT === 'testing') {
					$error_message .= ' Error: ' . $e->getMessage() . ' (File: ' . basename($e->getFile()) . ' Line: ' . $e->getLine() . ')';
				}
				echo json_encode(array('status' => 'error', 'message' => $error_message));
				exit;
			}
			
			$this->session->set_flashdata('error', 'Terjadi kesalahan saat memproses checkout! Silakan coba lagi.');
			redirect('checkout');
		} catch (Throwable $e) {
			// Tangkap semua error termasuk fatal error (PHP 7+)
			log_message('error', 'Checkout proses fatal error: ' . $e->getMessage());
			log_message('error', 'Checkout proses trace: ' . $e->getTraceAsString());
			log_message('error', 'Checkout proses file: ' . $e->getFile() . ' line: ' . $e->getLine());
			
			// Rollback transaction jika ada
			try {
				if ($this->db->trans_status() !== FALSE) {
					$this->db->trans_rollback();
				}
			} catch (Exception $rollback_ex) {
				log_message('error', 'Failed to rollback: ' . $rollback_ex->getMessage());
			}
			
			if($is_ajax) {
				if (ob_get_level()) {
					ob_clean();
				}
				header('Content-Type: application/json; charset=utf-8');
				$error_message = 'Terjadi kesalahan fatal saat memproses checkout.';
				// Di development, tampilkan error detail
				if (ENVIRONMENT === 'development' || ENVIRONMENT === 'testing') {
					$error_message .= ' Error: ' . $e->getMessage() . ' (File: ' . basename($e->getFile()) . ' Line: ' . $e->getLine() . ')';
				}
				echo json_encode(array('status' => 'error', 'message' => $error_message));
				exit;
			}
			
			$this->session->set_flashdata('error', 'Terjadi kesalahan fatal saat memproses checkout! Silakan coba lagi.');
			redirect('checkout');
		}
	}

	public function payment()
	{
		// Cek apakah user sudah login
		if(!$this->session->userdata('logged_in')) {
			$this->session->set_flashdata('error', 'Silakan login terlebih dahulu!');
			redirect('akun');
		}

		$snap_token = $this->session->userdata('snap_token');
		$order_id = $this->session->userdata('order_id');
		
		if(empty($snap_token) || empty($order_id)) {
			$this->session->set_flashdata('error', 'Sesi pembayaran tidak valid!');
			redirect('checkout');
		}

		$data['title'] = 'Pembayaran - Marketplace';
		$data['nama_customer'] = $this->session->userdata('nama_customer');
		$data['logged_in'] = $this->session->userdata('logged_in');
		$data['snap_token'] = $snap_token;
		$data['client_key'] = $this->midtrans->get_client_key();
		$data['is_production'] = $this->midtrans->is_production();
		
		$this->load->view('header', $data);
		$this->load->view('payment', $data);
		$this->load->view('footer');
	}

	public function finish()
	{
		// Hapus session snap token
		$this->session->unset_userdata('snap_token');
		$this->session->unset_userdata('order_id');
		
		$order_id = $this->input->get('order_id');
		
		if($order_id) {
			// Cari transaksi berdasarkan kode_transaksi (karena order_id disimpan sebagai kode_transaksi)
			$this->db->where('kode_transaksi', $order_id);
			$transaksi = $this->db->get('transaksi')->row();
			
			if($transaksi) {
				// Cek status transaksi dari Midtrans untuk memastikan pembayaran berhasil
				try {
					$midtrans_status = $this->midtrans->get_status($order_id);
					
					if($midtrans_status && isset($midtrans_status['transaction_status'])) {
						$transaction_status = $midtrans_status['transaction_status'];
						$fraud_status = isset($midtrans_status['fraud_status']) ? $midtrans_status['fraud_status'] : '';
						
						// Update status berdasarkan status dari Midtrans
						$status = 'pending';
						if($transaction_status == 'settlement' || $transaction_status == 'capture') {
							if($fraud_status == 'accept' || empty($fraud_status)) {
								$status = 'dibayar';
							}
						} else if($transaction_status == 'pending') {
							$status = 'pending';
						} else if($transaction_status == 'deny' || $transaction_status == 'expire' || $transaction_status == 'cancel') {
							$status = 'dibatalkan';
						}
						
						// Update status transaksi
						$this->db->where('id_transaksi', $transaksi->id_transaksi);
						$this->db->update('transaksi', array('status_transaksi' => $status));
						
						log_message('info', 'Checkout finish: Order ID ' . $order_id . ' - Status updated to: ' . $status);
					}
				} catch (Exception $e) {
					log_message('error', 'Failed to get Midtrans status in finish(): ' . $e->getMessage());
					// Tetap lanjutkan meskipun gagal cek status
				}
				
				$this->session->set_flashdata('success', 'Pembayaran berhasil! Terima kasih atas pembelian Anda.');
				redirect('transaksi/detail/'.$transaksi->id_transaksi);
			}
		}
		
		$this->session->set_flashdata('info', 'Terima kasih! Kami akan memproses pesanan Anda.');
		redirect('transaksi');
	}

	public function unfinish()
	{
		// Hapus session snap token
		$this->session->unset_userdata('snap_token');
		$this->session->unset_userdata('order_id');
		
		$order_id = $this->input->get('order_id');
		
		if($order_id) {
			// Cari transaksi berdasarkan kode_transaksi (karena order_id disimpan sebagai kode_transaksi)
			$this->db->where('kode_transaksi', $order_id);
			$transaksi = $this->db->get('transaksi')->row();
			
			if($transaksi) {
				$this->session->set_flashdata('warning', 'Pembayaran belum selesai. Silakan selesaikan pembayaran Anda.');
				redirect('transaksi/detail/'.$transaksi->id_transaksi);
			}
		}
		
		$this->session->set_flashdata('warning', 'Pembayaran belum selesai. Silakan selesaikan pembayaran Anda.');
		redirect('transaksi');
	}

	public function error()
	{
		// Hapus session snap token
		$this->session->unset_userdata('snap_token');
		$this->session->unset_userdata('order_id');
		
		$order_id = $this->input->get('order_id');
		
		if($order_id) {
			// Cari transaksi berdasarkan kode_transaksi (karena order_id disimpan sebagai kode_transaksi)
			$this->db->where('kode_transaksi', $order_id);
			$transaksi = $this->db->get('transaksi')->row();
			
			if($transaksi) {
				// Update status menjadi dibatalkan - gunakan status_transaksi bukan status
				$this->db->where('id_transaksi', $transaksi->id_transaksi);
				$this->db->update('transaksi', array('status_transaksi' => 'dibatalkan'));
				
				$this->session->set_flashdata('error', 'Pembayaran gagal atau dibatalkan.');
				redirect('transaksi/detail/'.$transaksi->id_transaksi);
			}
		}
		
		$this->session->set_flashdata('error', 'Pembayaran gagal atau dibatalkan.');
		redirect('transaksi');
	}

	public function notification()
	{
		// Handler untuk webhook notification dari Midtrans
		$json = file_get_contents('php://input');
		$notification = json_decode($json, true);
		
		if($notification) {
			$order_id = isset($notification['order_id']) ? $notification['order_id'] : '';
			$transaction_status = isset($notification['transaction_status']) ? $notification['transaction_status'] : '';
			$fraud_status = isset($notification['fraud_status']) ? $notification['fraud_status'] : '';
			$gross_amount = isset($notification['gross_amount']) ? $notification['gross_amount'] : 0;
			$signature_key = isset($notification['signature_key']) ? $notification['signature_key'] : '';
			$status_code = isset($notification['status_code']) ? $notification['status_code'] : '';
			
			// Verifikasi signature
			$is_valid = $this->midtrans->verify_signature($order_id, $status_code, $gross_amount, $signature_key);
			
			if($is_valid) {
				// Cari transaksi berdasarkan kode_transaksi (karena order_id disimpan sebagai kode_transaksi)
				$this->db->where('kode_transaksi', $order_id);
				$transaksi = $this->db->get('transaksi')->row();
				
				if($transaksi) {
					// Update status transaksi berdasarkan status dari Midtrans
					// Gunakan status_transaksi dengan nilai enum yang sesuai: 'pending', 'dibayar', 'diproses', 'dikirim', dll
					$status = 'pending';
					
					if($transaction_status == 'settlement' || $transaction_status == 'capture') {
						// Settlement atau capture berarti pembayaran berhasil
						// Tidak perlu cek fraud_status untuk settlement, karena sudah verified
						$status = 'dibayar';
					} else if($transaction_status == 'pending') {
						$status = 'pending';
					} else if($transaction_status == 'deny' || $transaction_status == 'expire' || $transaction_status == 'cancel') {
						$status = 'dibatalkan';
					}
					
					// Update transaksi - gunakan status_transaksi bukan status
					$update_data = array('status_transaksi' => $status);
					
					// Cek kolom yang ada sebelum update
					if ($this->db->field_exists('payment_type', 'transaksi')) {
						$update_data['payment_type'] = isset($notification['payment_type']) ? $notification['payment_type'] : '';
					}
					if ($this->db->field_exists('payment_method', 'transaksi')) {
						$update_data['payment_method'] = isset($notification['payment_method']) ? $notification['payment_method'] : '';
					}
					
					$this->db->where('id_transaksi', $transaksi->id_transaksi);
					$update_result = $this->db->update('transaksi', $update_data);
					
					// Log notification dengan detail
					log_message('info', 'Midtrans Notification: Order ID ' . $order_id . ' - Transaction Status: ' . $transaction_status . ' - Updated to: ' . $status . ' - Update result: ' . ($update_result ? 'success' : 'failed'));
					
					if (!$update_result) {
						$db_error = $this->db->error();
						log_message('error', 'Failed to update transaction status: ' . print_r($db_error, true));
					}
				} else {
					log_message('warning', 'Midtrans Notification: Transaction not found for Order ID: ' . $order_id);
				}
			}
		}
		
		// Return 200 OK untuk Midtrans
		http_response_code(200);
		echo "OK";
	}
}

