<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Midtrans {
	
	private $server_key;
	private $is_production;
	private $client_key;
	
	public function __construct() {
		// Load konfigurasi Midtrans dari file config
		$CI =& get_instance();
		$CI->load->config('midtrans');
		
		$this->is_production = $CI->config->item('midtrans_is_production');
		
		if($this->is_production) {
			// Production Mode
			$this->server_key = $CI->config->item('midtrans_server_key_production');
			$this->client_key = $CI->config->item('midtrans_client_key_production');
		} else {
			// Sandbox Mode (Testing)
			$this->server_key = $CI->config->item('midtrans_server_key_sandbox');
			$this->client_key = $CI->config->item('midtrans_client_key_sandbox');
		}
		
		// Validasi bahwa keys tidak kosong
		if(empty($this->server_key) || empty($this->client_key)) {
			log_message('error', 'Midtrans: Server Key atau Client Key kosong! Pastikan konfigurasi sudah benar.');
		}
		
		// Log untuk debugging (jangan log key lengkap untuk security)
		log_message('info', 'Midtrans initialized - Production: ' . ($this->is_production ? 'YES' : 'NO') . ', Server Key: ' . substr($this->server_key, 0, 15) . '...');
	}
	
	/**
	 * Get Client Key
	 */
	public function get_client_key() {
		return $this->client_key;
	}
	
	/**
	 * Check if production mode
	 */
	public function is_production() {
		return $this->is_production;
	}
	
	/**
	 * Membuat transaksi Snap Token untuk Midtrans
	 */
	public function get_snap_token($params) {
		$endpoint = $this->is_production 
			? 'https://app.midtrans.com/snap/v1/transactions'
			: 'https://app.sandbox.midtrans.com/snap/v1/transactions';
		
		$curl = curl_init();
		
		// Konfigurasi SSL Certificate
		// Cek beberapa path CA bundle yang mungkin ada
		$ca_bundle_paths = array(
			'C:\\xampp\\apache\\bin\\curl-ca-bundle.crt',
			'C:\\xampp\\php\\extras\\ssl\\cacert.pem',
			'C:\\xampp\\php\\cacert.pem',
			getcwd() . '\\cacert.pem',
			__DIR__ . '\\..\\..\\cacert.pem'
		);
		
		$ca_bundle_found = false;
		$ca_bundle_path = null;
		
		// Cari CA bundle yang ada
		foreach ($ca_bundle_paths as $path) {
			if (file_exists($path) && is_readable($path)) {
				$ca_bundle_path = $path;
				$ca_bundle_found = true;
				break;
			}
		}
		
		// Setup CURL options
		// Validasi server_key sebelum digunakan
		if(empty($this->server_key)) {
			log_message('error', 'Midtrans: Server Key is empty!');
			throw new Exception('Server Key Midtrans tidak ditemukan. Pastikan konfigurasi sudah benar.');
		}
		
		// Setup Authorization header
		$auth_header = 'Basic ' . base64_encode($this->server_key . ':');
		
		$curl_options = array(
			CURLOPT_URL => $endpoint,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 30, // Set timeout 30 detik
			CURLOPT_CONNECTTIMEOUT => 10, // Connection timeout 10 detik
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => 'POST',
			CURLOPT_POSTFIELDS => json_encode($params),
			CURLOPT_HTTPHEADER => array(
				'Accept: application/json',
				'Content-Type: application/json',
				'Authorization: ' . $auth_header
			),
		);
		
		// Log untuk debugging (jangan log key lengkap)
		log_message('info', 'Midtrans Request - Endpoint: ' . $endpoint . ', Server Key: ' . substr($this->server_key, 0, 15) . '...');
		
		// Set SSL options
		if ($ca_bundle_found) {
			// Gunakan CA bundle jika ditemukan
			$curl_options[CURLOPT_CAINFO] = $ca_bundle_path;
			$curl_options[CURLOPT_SSL_VERIFYPEER] = true;
			$curl_options[CURLOPT_SSL_VERIFYHOST] = 2;
			log_message('info', 'Using CA bundle: ' . $ca_bundle_path);
		} else {
			// Untuk development/testing, disable SSL verification jika CA bundle tidak ditemukan
			// PENTING: Untuk production, sebaiknya download dan set CA bundle yang benar
			$curl_options[CURLOPT_SSL_VERIFYPEER] = false;
			$curl_options[CURLOPT_SSL_VERIFYHOST] = false;
			log_message('warning', 'CA bundle not found. SSL verification disabled. This is OK for development but not recommended for production.');
		}
		
		curl_setopt_array($curl, $curl_options);
		
		$response = curl_exec($curl);
		$http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		$curl_error = curl_error($curl);
		curl_close($curl);
		
		// Log error jika ada
		if (!empty($curl_error)) {
			log_message('error', 'Midtrans CURL Error: ' . $curl_error);
			throw new Exception('Gagal menghubungi server Midtrans: ' . $curl_error);
		}
		
		if ($http_code == 201) {
			$result = json_decode($response, true);
			if (isset($result['token']) && !empty($result['token'])) {
				return $result['token'];
			} else {
				log_message('error', 'Midtrans Response: Token tidak ditemukan dalam response. Response: ' . $response);
				throw new Exception('Token tidak ditemukan dalam response Midtrans');
			}
		} else {
			$error_data = json_decode($response, true);
			$error_message = 'HTTP ' . $http_code;
			if (isset($error_data['error_messages']) && is_array($error_data['error_messages'])) {
				$error_message .= ': ' . implode(', ', $error_data['error_messages']);
			} else if (isset($error_data['message'])) {
				$error_message .= ': ' . $error_data['message'];
			} else {
				$error_message .= ': ' . $response;
			}
			log_message('error', 'Midtrans API Error: ' . $error_message);
			throw new Exception('Gagal membuat transaksi Midtrans. ' . $error_message);
		}
	}
	
	/**
	 * Verifikasi status transaksi dari Midtrans
	 */
	public function get_status($order_id) {
		$endpoint = $this->is_production
			? 'https://api.midtrans.com/v2/' . $order_id . '/status'
			: 'https://api.sandbox.midtrans.com/v2/' . $order_id . '/status';
		
		$curl = curl_init();
		
		// Konfigurasi SSL Certificate (sama seperti get_snap_token)
		$ca_bundle_paths = array(
			'C:\\xampp\\apache\\bin\\curl-ca-bundle.crt',
			'C:\\xampp\\php\\extras\\ssl\\cacert.pem',
			'C:\\xampp\\php\\cacert.pem',
			getcwd() . '\\cacert.pem',
			__DIR__ . '\\..\\..\\cacert.pem'
		);
		
		$ca_bundle_found = false;
		$ca_bundle_path = null;
		
		foreach ($ca_bundle_paths as $path) {
			if (file_exists($path) && is_readable($path)) {
				$ca_bundle_path = $path;
				$ca_bundle_found = true;
				break;
			}
		}
		
		$curl_options = array(
			CURLOPT_URL => $endpoint,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 30,
			CURLOPT_CONNECTTIMEOUT => 10,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => 'GET',
			CURLOPT_HTTPHEADER => array(
				'Accept: application/json',
				'Authorization: Basic ' . base64_encode($this->server_key . ':')
			),
		);
		
		if ($ca_bundle_found) {
			$curl_options[CURLOPT_CAINFO] = $ca_bundle_path;
			$curl_options[CURLOPT_SSL_VERIFYPEER] = true;
			$curl_options[CURLOPT_SSL_VERIFYHOST] = 2;
		} else {
			$curl_options[CURLOPT_SSL_VERIFYPEER] = false;
			$curl_options[CURLOPT_SSL_VERIFYHOST] = false;
		}
		
		curl_setopt_array($curl, $curl_options);
		
		$response = curl_exec($curl);
		$http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		$curl_error = curl_error($curl);
		curl_close($curl);
		
		if (!empty($curl_error)) {
			log_message('error', 'Midtrans get_status CURL Error: ' . $curl_error);
		}
		
		if ($http_code == 200) {
			return json_decode($response, true);
		} else {
			log_message('error', 'Midtrans Status Error: ' . $response);
			return false;
		}
	}
	
	/**
	 * Verifikasi signature dari webhook Midtrans
	 */
	public function verify_signature($order_id, $status_code, $gross_amount, $signature_key) {
		$expected_signature = hash('sha512', $order_id . $status_code . $gross_amount . $this->server_key);
		return hash_equals($expected_signature, $signature_key);
	}

	/**
	 * Cancel transaction in Midtrans
	 */
	public function cancel($order_id) {
		$endpoint = $this->is_production
			? 'https://api.midtrans.com/v2/' . $order_id . '/cancel'
			: 'https://api.sandbox.midtrans.com/v2/' . $order_id . '/cancel';
		
		$curl = curl_init();
		
		// Konfigurasi SSL Certificate (sama seperti get_snap_token)
		$ca_bundle_paths = array(
			'C:\\xampp\\apache\\bin\\curl-ca-bundle.crt',
			'C:\\xampp\\php\\extras\\ssl\\cacert.pem',
			'C:\\xampp\\php\\cacert.pem',
			getcwd() . '\\cacert.pem',
			__DIR__ . '\\..\\..\\cacert.pem'
		);
		
		$ca_bundle_found = false;
		$ca_bundle_path = null;
		
		foreach ($ca_bundle_paths as $path) {
			if (file_exists($path) && is_readable($path)) {
				$ca_bundle_path = $path;
				$ca_bundle_found = true;
				break;
			}
		}
		
		$curl_options = array(
			CURLOPT_URL => $endpoint,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 30,
			CURLOPT_CONNECTTIMEOUT => 10,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => 'POST',
			CURLOPT_HTTPHEADER => array(
				'Accept: application/json',
				'Content-Type: application/json',
				'Authorization: Basic ' . base64_encode($this->server_key . ':')
			),
		);
		
		if ($ca_bundle_found) {
			$curl_options[CURLOPT_CAINFO] = $ca_bundle_path;
			$curl_options[CURLOPT_SSL_VERIFYPEER] = true;
			$curl_options[CURLOPT_SSL_VERIFYHOST] = 2;
		} else {
			$curl_options[CURLOPT_SSL_VERIFYPEER] = false;
			$curl_options[CURLOPT_SSL_VERIFYHOST] = false;
		}
		
		curl_setopt_array($curl, $curl_options);
		
		$response = curl_exec($curl);
		$http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		$curl_error = curl_error($curl);
		curl_close($curl);
		
		if (!empty($curl_error)) {
			log_message('error', 'Midtrans Cancel CURL Error: ' . $curl_error);
			throw new Exception('Gagal menghubungi server Midtrans untuk pembatalan: ' . $curl_error);
		}
		
		$result = json_decode($response, true);
		
		// Success codes for cancel usually include 200 or 409 (already cancelled)
		if ($http_code == 200) {
			return $result;
		} else {
			// Jika error 412 (Precondition Failed), biasanya karena status tidak bisa di-cancel (sudah settlement/expire)
			// Kita kembalikan result apa adanya agar caller bisa memproses
			log_message('warning', 'Midtrans Cancel Failed (HTTP ' . $http_code . '): ' . $response);
			return $result;
		}
	}
}

