<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Mongkir extends CI_Model {

	private $api_key;
	private $base_url;
	private $is_komerce = false;

	public function __construct()
	{
		parent::__construct();
		// Load config centralized
		$this->load->config('rajaongkir', TRUE);
		$this->api_key = $this->config->item('rajaongkir_api_key', 'rajaongkir');
		$this->base_url = $this->config->item('rajaongkir_base_url', 'rajaongkir');

		// Detect Komerce API Key (starts with v1 or longer than 32 chars)
		// Komerce keys are usually base64-like strings, while RajaOngkir standard keys are 32 hex chars.
		if (!empty($this->api_key) && (strlen($this->api_key) > 32 || substr($this->api_key, 0, 2) === 'v1')) {
			$this->is_komerce = true;
			// Use Komerce endpoint
			$this->base_url = 'https://rajaongkir.komerce.id/api/v1';
			log_message('info', 'Mongkir: Detected Komerce API Key. Switching to Komerce Endpoint: ' . $this->base_url);
		}

		if (empty($this->api_key)) {
			log_message('error', 'Mongkir: RajaOngkir API key belum diset di application/config/rajaongkir.php');
		}
	}

	/**
	 * Helper method untuk melakukan cURL request
	 */
	private function curl_request($url, $method = 'GET', $postfields = null)
	{
		$curl = curl_init();

		$options = array(
			CURLOPT_URL => $url,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => "",
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 30,
			CURLOPT_CONNECTTIMEOUT => 30,
			CURLOPT_SSL_VERIFYPEER => false, // Bypass SSL for local dev
			CURLOPT_SSL_VERIFYHOST => false,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => $method,
		);

		// Headers
		$headers = array('key: ' . $this->api_key);

		if ($method == 'POST' && $postfields) {
			$options[CURLOPT_POSTFIELDS] = $postfields;
			$headers[] = 'content-type: application/x-www-form-urlencoded';
		}

		$options[CURLOPT_HTTPHEADER] = $headers;

		curl_setopt_array($curl, $options);

		$response = curl_exec($curl);
		$err = curl_error($curl);
		$http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		curl_close($curl);

		if ($err) {
			$msg = 'Raja Ongkir CURL Error: ' . $err . ' | URL: ' . $url;
			log_message('error', $msg);
			throw new Exception($msg);
		}

		if ($http_code == 0) {
			$msg = 'Raja Ongkir: Tidak dapat menghubungi API (HTTP 0). Periksa koneksi internet. URL: ' . $url;
			log_message('error', $msg);
			throw new Exception($msg);
		}

		$decoded = json_decode($response, TRUE);
		if ($decoded === null) {
			$msg = 'Raja Ongkir: Gagal decode JSON response. Raw: ' . substr($response, 0, 1000);
			log_message('error', $msg);
			throw new Exception($msg);
		}

		// Komerce Error Handling
		if ($this->is_komerce) {
			if (isset($decoded['meta']['code']) && $decoded['meta']['code'] != 200) {
				$msg = 'Komerce API Error: ' . (isset($decoded['meta']['message']) ? $decoded['meta']['message'] : 'Unknown error');
				log_message('error', $msg);
				throw new Exception($msg);
			}
		} else {
			// Standard RajaOngkir Error Handling
			if ($http_code == 410) {
				$msg = 'Raja Ongkir Error 410: Endpoint tidak aktif. Pastikan Anda menggunakan API Key RajaOngkir yang valid (Starter/Basic/Pro) dari rajaongkir.com, bukan key dari layanan migrasi/komerce.';
				log_message('error', $msg);
				throw new Exception($msg);
			}

			if ($http_code != 200) {
				$msg = 'Raja Ongkir HTTP Error ' . $http_code . ': ' . $response . ' | URL: ' . $url;
				log_message('error', $msg);
				throw new Exception($msg);
			}

			if (isset($decoded['rajaongkir']['status']['code']) && $decoded['rajaongkir']['status']['code'] != 200) {
				$descr = isset($decoded['rajaongkir']['status']['description']) ? $decoded['rajaongkir']['status']['description'] : 'Unknown';
				$msg = 'Raja Ongkir API Error: ' . $descr;
				log_message('error', $msg);
				throw new Exception($msg);
			}
		}

		return $decoded;
	}

	/**
	 * Menghitung ongkir
	 */
	public function hitung_ongkir($origin, $destination, $weight, $courier = 'jne')
	{
		if (empty($origin) || empty($destination) || empty($weight)) {
			throw new Exception('Parameter ongkir tidak lengkap');
		}

		if ($weight < 1) {
			$weight = 1000;
		}

		$postfields = "origin=$origin&destination=$destination&weight=$weight&courier=$courier";

		if ($this->is_komerce) {
			// Komerce Endpoint
			// Note: Assuming 'calculate/domestic-cost' works for City IDs or generic inputs.
			// If strictly District IDs are required, this might need adjustment based on input data.
			$url = $this->base_url . '/calculate/domestic-cost'; 
			
			$result = $this->curl_request($url, 'POST', $postfields);

			$formatted = array();
			// Parse Komerce Response Structure
			// Expected: { meta: {...}, data: [ { name: "Lion", code: "lion", service: "REG", cost: 7000, etd: "1-2" }, ... ] }
			if (isset($result['data']) && is_array($result['data'])) {
				foreach ($result['data'] as $cost) {
					$formatted[] = array(
						'courier' => isset($cost['code']) ? strtoupper($cost['code']) : (isset($cost['name']) ? strtoupper($cost['name']) : strtoupper($courier)),
						'service' => isset($cost['service']) ? $cost['service'] : '',
						'description' => isset($cost['description']) ? $cost['description'] : '',
						'cost' => isset($cost['cost']) ? (int)$cost['cost'] : 0,
						'etd' => isset($cost['etd']) ? $cost['etd'] : '',
						'note' => ''
					);
				}
				return $formatted;
			} else {
			    // Try fallback to standard structure if Komerce wraps standard response?
			    // Unlikely, but good to be safe or log warning.
			    log_message('error', 'Mongkir: Komerce response format unrecognized. Raw: ' . json_encode($result));
			}
			
		} else {
			// Standard RajaOngkir Endpoint
			$url = $this->base_url . '/cost';
			$result = $this->curl_request($url, 'POST', $postfields);

			if (isset($result['rajaongkir']['results'][0])) {
				$data = $result['rajaongkir']['results'][0];
				$formatted = array();

				if (isset($data['costs']) && is_array($data['costs'])) {
					foreach ($data['costs'] as $cost) {
						$formatted[] = array(
							'courier' => isset($data['name']) ? $data['name'] : strtoupper($courier),
							'service' => isset($cost['service']) ? $cost['service'] : '',
							'description' => isset($cost['description']) ? $cost['description'] : '',
							'cost' => isset($cost['cost'][0]['value']) ? (int)$cost['cost'][0]['value'] : 0,
							'etd' => isset($cost['cost'][0]['etd']) ? $cost['cost'][0]['etd'] : '',
							'note' => isset($cost['cost'][0]['note']) ? $cost['cost'][0]['note'] : ''
						);
					}
				}
				return $formatted;
			}
		}
		
		return array();
	}

	/**
	 * Validasi data ongkir
	 */
	public function validasi_ongkir($origin, $destination, $weight, $courier)
	{
		if (empty($origin) || !is_numeric($origin)) return false;
		if (empty($destination) || !is_numeric($destination)) return false;
		if (empty($weight) || !is_numeric($weight) || $weight < 1) return false;
		
		// Expanded courier list for Komerce
		$kurir_valid = array('jne', 'pos', 'tiki', 'sicepat', 'jnt', 'lion', 'ninja', 'anteraja', 'ide', 'sap');
		if (!in_array(strtolower($courier), $kurir_valid)) return false;
		
		return true;
	}

	// Keep existing methods for province/city but wrap with check
	public function get_all_provinsi()
	{
		if ($this->is_komerce) {
		    // Komerce Province Endpoint: /destination/province
		    $url = $this->base_url . '/destination/province';
		    try {
		        $result = $this->curl_request($url);
		        if (isset($result['data']) && is_array($result['data'])) {
		            // Load Standard Data for mapping
		            $this->load->helper('indonesia_data');
		            $standard_provinces = get_provinsi_static();
		            
		            $formatted = array();
		            foreach ($result['data'] as $p) {
		                $komerce_name = isset($p['name']) ? strtoupper(trim($p['name'])) : '';
		                $standard_id = isset($p['id']) ? $p['id'] : ''; // Default to Komerce ID
		                
		                // Find Standard ID by Name
		                foreach ($standard_provinces as $sp) {
		                    $standard_name = strtoupper(trim($sp['province']));
		                    // Check if names match or are very similar
		                    if ($komerce_name === $standard_name || 
		                        strpos($komerce_name, $standard_name) !== false || 
		                        strpos($standard_name, $komerce_name) !== false) {
		                        $standard_id = $sp['province_id'];
		                        break;
		                    }
		                    
		                    // Specific manual fixes for known discrepancies
		                    if ($komerce_name == 'DKI JAKARTA' && $standard_name == 'DKI JAKARTA') $standard_id = '6';
		                    if (strpos($komerce_name, 'YOGYAKARTA') !== false && strpos($standard_name, 'YOGYAKARTA') !== false) $standard_id = '5';
		                }
		                
		                $formatted[] = array(
		                    'province_id' => $standard_id,
		                    'province' => isset($p['name']) ? $p['name'] : ''
		                );
		            }
		            return $formatted;
		        }
		    } catch (Exception $e) {
		        log_message('error', 'Mongkir: Komerce Province Error: ' . $e->getMessage());
		    }
		    // Fallback static
		    $this->load->helper('indonesia_data');
		    return get_provinsi_static();
		}
		
		// ... existing standard logic ...
		try {
			$result = $this->curl_request($this->base_url . '/province');
			if (isset($result['rajaongkir']['results'])) {
				return $result['rajaongkir']['results'];
			}
			throw new Exception('Invalid response');
		} catch (Exception $e) {
			$this->load->helper('indonesia_data');
			return get_provinsi_static();
		}
	}

	public function get_kota_by_provinsi($id_provinsi)
	{
	    if (empty($id_provinsi)) throw new Exception('Provinsi tidak valid');

		if ($this->is_komerce) {
		    // Komerce City Endpoint is currently returning 404 for standard endpoints.
		    // Fallback to static data immediately to ensure functionality.
		    $this->load->helper('indonesia_data');
		    $kota_static = get_kota_by_provinsi_static($id_provinsi);
		    
		    if (!empty($kota_static)) {
		        return $kota_static;
		    } else {
		        // If static data is missing for this province, try to fetch from API anyway (best effort)
		        // Try multiple potential endpoints if one fails? No, let's stick to one or return empty.
		        // We return empty to avoid breaking the page with uncaught exceptions, 
		        // but log it so we know we need more static data.
		        log_message('error', 'Mongkir: No static city data for province ID ' . $id_provinsi);
		        return array();
		    }
		}

		// ... existing standard logic ...
		try {
			$result = $this->curl_request($this->base_url . '/city?province=' . $id_provinsi);
			if (isset($result['rajaongkir']['results'])) {
				return $result['rajaongkir']['results'];
			}
			throw new Exception('Invalid response');
		} catch (Exception $e) {
			$this->load->helper('indonesia_data');
			$kota_static = get_kota_by_provinsi_static($id_provinsi);
			return !empty($kota_static) ? $kota_static : array();
		}
	}
	
	// Legacy aliases
	public function switch_to_basic() { return false; }
	public function tampil_distrik() { return $this->get_kota_by_provinsi(null); } // Simplified
	public function detail_distrik($city_id) { return false; }
	public function biaya($origin, $destination, $weight, $courier = 'jne') {
		return $this->hitung_ongkir($origin, $destination, $weight, $courier);
	}
}
